import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit

# Modèle exponentiel croissant
def modele_exp(t, A, tau, C):
    return A * (1 - np.exp(-t / tau)) + C

# Fonction principale : fit + zoom + tracé
def fit_and_plot_zoom(csv_file, zoom_start, zoom_end):
    # Lecture des données (colonnes temps et tension)
    data = pd.read_csv(csv_file, skiprows=18, header=None, usecols=[3, 4], names=["Time (s)", "Voltage (V)"])
    data["Voltage (V)"] = data["Voltage (V)"] / 10  # Conversion en volts

    # Sélection de la zone zoomée autour de la montée
    zoom_data = data[(data["Time (s)"] >= zoom_start) & (data["Time (s)"] <= zoom_end)]

    # Préparation pour le fit
    t = zoom_data["Time (s)"].values
    t_fit = t - t[0]  # Recentrage du temps
    y = zoom_data["Voltage (V)"].values

    # Filtrage entre 5% et 95% de la montée pour garder la vraie croissance exponentielle
    Vmin, Vmax = y.min(), y.max()
    y_norm = (y - Vmin) / (Vmax - Vmin)
    mask = (y_norm >= 0.01) & (y_norm <= 0.98)
    t_fit = t_fit[mask]
    y = y[mask]

    # Ajustement du modèle exponentiel
    try:
        params, _ = curve_fit(modele_exp, t_fit, y, p0=[5, 1e-5, 0])
        t_model = np.linspace(t_fit.min(), t_fit.max(), 500)
        y_model = modele_exp(t_model, *params)
        fit_label = f"Fit expo croissante : τ = {params[1]*1e6:.1f} µs"

        # Affichage des paramètres
        print("Paramètres du fit :")
        print(f"  A = {params[0]:.3f} V")
        print(f"  τ = {params[1]*1e6:.2f} µs")
        print(f"  C = {params[2]:.3f} V")

    except Exception as e:
        print("Erreur lors du fit :", e)
        return

    # Tracé zoomé avec le fit superposé
    plt.figure(figsize=(10, 5))
    plt.plot(zoom_data["Time (s)"], zoom_data["Voltage (V)"], 'b-', label="Signal (zoomé)")
    plt.plot(t_model + t[0], y_model, 'r--', linewidth=2, label=fit_label)
    plt.xlabel("Temps (s)")
    plt.ylabel("Tension (V)")
    plt.title("Zoom sur la montée avec fit exponentiel Cl36")
    plt.legend()
    plt.grid(True)
    plt.tight_layout()
    plt.show()

# === UTILISATION ===
csv_file_path = "/Users/lara/Desktop/L3_physique/S6/projet /données oscillo /ALL0007/F0007CH2.CSV"  # Chemin du fichier
fit_and_plot_zoom(csv_file_path, zoom_start=-0.00002, zoom_end=0.00005)
